import { Ref } from 'vue';
import { useRoute } from 'vue-router';
import { ViewPropsBase, ViewStateBase, UIBase, IParam, UIUtil } from '@core';

/**
 * @description 视图基类
 * @export
 * @class ViewBase
 */
export class ViewBase {

  /**
   * @description 视图状态
   * @type {ViewStateBase}
   * @memberof ViewBase
   */
  public viewState: ViewStateBase;

  /**
   * 界面行为服务
   *
   * @type {(IParam | undefined)}
   * @memberof ViewBase
   */
  public appUIService: IParam | undefined;

  /**
   * 数据服务
   *
   * @type {(IParam | undefined)}
   * @memberof ViewBase
   */
  public appDataService: IParam | undefined;

  /**
   * Creates an instance of ViewBase.
   * @param {*} options 配置参数
   * @memberof ViewBase
   */
  constructor(options: any) {
    this.viewState = reactive(options);
  }

  /**
   * @description 根据props调整视图状态变量
   * @param {ViewPropsBase} props 传入的Props
   * @param {Function} emit 事件
   * @memberof ViewBase
   */
  protected handleState(props: ViewPropsBase, emit: Function) {
    // 只读属性处理：把props的属性的Ref赋给state，以维持响应性
    this.viewState.viewSubject = toRef(props, 'viewSubject') as any;
  }

  /**
   * 通知状态
   *
   * @param { tag: string, action: string, data: any } { tag, action, data }
   * @memberof ViewEngine
   */
  public next({ tag, action, data }: { tag: string, action: string, data: any }): void {
    const { viewSubject } = this.viewState;
    viewSubject.next({ tag: tag, action: action, data: data })
  }

  /**
   * @description 处理视图导航参数
   * 
   * @param props 输入属性
   * @param context 应用上下文
   * @param viewParams 视图参数
   */
  private handleViewContextParams(props: ViewPropsBase, context: Ref<IParam | undefined>, viewParams: Ref<IParam | undefined>) {
    if (!context.value) context.value = {};
    if (!viewParams.value) viewParams.value = {};
    const { appViewNavContexts, appViewNavParams } = this.viewState;
    if (Object.is(props.openType, 'ROUTE')) {
      // 应用上下文
      const appContext = App.getAppData();
      Object.assign(context.value, appContext);
      // 视图应用上下文
      const pageContext = {};
      const routeParams = useRoute().params;
      if (routeParams && (Object.keys(routeParams).length > 0)) {
        Object.keys(routeParams).forEach((key: string) => {
          if (routeParams[key]) {
            Object.assign(pageContext, { [key]: decodeURIComponent(routeParams[key] as string) });
          }
        })
      }
      Object.assign(context.value, pageContext);
      // 视图参数
      const pageViewParams = {};
      const routeQuerys = useRoute().query;
      if (routeQuerys && (Object.keys(routeQuerys).length > 0)) {
        Object.keys(routeQuerys).forEach((key: string) => {
          if (routeQuerys[key]) {
            Object.assign(pageViewParams, { [key]: routeQuerys[key] });
          }
        })
      }
      Object.assign(viewParams.value, pageViewParams);
      // 视图自定义应用上下文
      const pageCustomContext = UIUtil.computedNavData(null, context.value, viewParams.value, appViewNavContexts);
      Object.assign(context.value, pageCustomContext);
      // 视图自定义视图参数
      const pageCustomViewParams = UIUtil.computedNavData(null, context.value, viewParams.value, appViewNavParams);
      Object.assign(viewParams.value, pageCustomViewParams);
    } else {
      // 视图自定义应用上下文
      const customContext = UIUtil.computedNavData(null, context.value, viewParams.value, appViewNavContexts);
      Object.assign(context.value, customContext);
      // 视图自定义视图参数
      const pageCustomViewParams = UIUtil.computedNavData(null, context.value, viewParams.value, appViewNavParams);
      Object.assign(viewParams.value, pageCustomViewParams);
    }
  }

  /**
   * @description 使用视图上下文参数逻辑块
   * 
   * @param {ViewPropsBase} props 传入的Props
   * @param {Function} emit 事件
   * @memberof ViewBase
   */
  public useViewContextParams(props: ViewPropsBase, emit: Function) {
    const context = UIBase.toOneWayRef(props, 'context');
    const viewParams = UIBase.toOneWayRef(props, 'viewParams');
    // 导航视图参数处理
    this.handleViewContextParams(props, context, viewParams);
    watch(context, (newVal: any, oldVal: any) => {
      this.handleViewContextParams(props, newVal, viewParams);
    });
    // 把Ref赋值到State上进行解包
    this.viewState.context = context;
    this.viewState.viewParams = viewParams;
    return { context, viewParams };
  }

  /**
   *@description 使用UI服务
   *
   * @param {ViewPropsBase} props
   * @param {Function} emit
   * @memberof ViewBase
   */
  public useUIService(props: ViewPropsBase, emit: Function) {
    const { appEntityName, context } = this.viewState;
    if (appEntityName) {
      App.getUIService(appEntityName.toLowerCase(), context).then((service: IParam) => {
        this.appUIService = service;
      })
    }
  }

  /**
   *@description 使用数据服务
   *
   * @param {ViewPropsBase} props
   * @param {Function} emit
   * @memberof ViewBase
   */
  public useDataService(props: ViewPropsBase, emit: Function) {
    const { appEntityName, context } = this.viewState;
    if (appEntityName) {
      App.getDataService(appEntityName.toLowerCase(), context).then((service: IParam) => {
        this.appDataService = service;
      })
    }
  }

  /**
   * @description 使用计数器服务
   * 
   * @param {ViewPropsBase} props 传入的Props
   * @param {Function} emit 事件
   * @memberof ViewBase
   */
  public useCounterService(props: ViewPropsBase, emit: Function) { }

  /**
   * @description 处理视图初始化
   *
   * @param {ViewPropsBase} props
   * @param {Function} emit
   * @memberof ViewBase
   */
  public useViewInit(props: ViewPropsBase, emit: Function) { }

  /**
   * @description 安装视图所有功能模块的方法
   * 
   * @param {ViewPropsBase} props 传入的Props
   * @param {Function} emit 事件
   * @memberof ViewBase
   */
  public moduleInstall(props: ViewPropsBase, emit: Function) {
    // 处理视图状态
    this.handleState(props, emit);
    // 处理视图导航参数
    this.useViewContextParams(props, emit);
    // 使用计数器服务
    this.useCounterService(props, emit);
    // 使用数据服务
    this.useDataService(props, emit);
    // 使用UI服务
    this.useUIService(props, emit);
    // 处理视图初始化
    this.useViewInit(props, emit);
    return {
      state: this.viewState
    };
  }
}
